﻿Imports Inventor



Module BRepExplain
    Private Structure FaceInfo
        Dim graphicsFace As GraphicsNode
        Dim normal As Vector
    End Structure

    ' Used to show the unique faces that make up a B-Rep body.
    Public Sub ExplodeBRep(ByVal offset As Double)
        Dim oPartDef As PartComponentDefinition = invApp.ActiveDocument.ComponentDefinition

        Dim globalTrans As Transaction = invApp.TransactionManager.StartGlobalTransaction(invApp.ActiveDocument, "Explode")

        Dim isForward As Boolean = True
        Dim numSteps As Integer = 30
        Dim oClientGraphics As ClientGraphics = Nothing
        Try
            oClientGraphics = oPartDef.ClientGraphicsCollection.Item("Explode")
        Catch ex As Exception
        End Try

        ' Client graphics already exist so delete them.
        If Not oClientGraphics Is Nothing Then
            oClientGraphics.Delete()
            isForward = False
        End If

        oClientGraphics = oPartDef.ClientGraphicsCollection.AddNonTransacting("Explode")

        ' Iterate through each face and create a client graphics version of it.
        Dim body As SurfaceBody = oPartDef.SurfaceBodies.Item(1)
        Dim faceList() As FaceInfo
        ReDim faceList(body.Faces.Count - 1)
        Dim faceCount As Integer = 0
        For Each oFace As Face In body.Faces
            ' Create default offset vector
            Dim offSetVec As Vector = tg.CreateVector(0, 0, 0)

            Dim oGraphicsNode As GraphicsNode = oClientGraphics.AddNode(1)
            oGraphicsNode.AddSurfaceGraphics(oFace)

            Dim bProcess As Boolean = True
            If oFace.SurfaceType = SurfaceTypeEnum.kCylinderSurface Then
                Dim bHasCircleEdge As Boolean = False
                For Each oEdge As Edge In oFace.Edges
                    If oEdge.GeometryType = CurveTypeEnum.kCircleCurve Then
                        bHasCircleEdge = True
                        Exit For
                    End If
                Next

                If bHasCircleEdge Then
                    bProcess = False
                End If
            End If

            If bProcess Then
                offSetVec = GetNormal(oFace)

                If isForward Then
                    offSetVec.ScaleBy(offset / numSteps)
                Else
                    Dim moveOffset As Vector = offSetVec.Copy()
                    moveOffset.ScaleBy(offset)

                    ' Move the face to the final position since we're undoing the explode.
                    Dim trans As Matrix = oGraphicsNode.Transformation
                    trans.SetTranslation(moveOffset)
                    oGraphicsNode.Transformation = trans

                    offSetVec.ScaleBy(-(offset / numSteps))
                End If
            End If

            faceList(faceCount).graphicsFace = oGraphicsNode
            faceList(faceCount).normal = offSetVec
            faceCount += 1
        Next

        oPartDef.SurfaceBodies.Item(1).Visible = False

        For i As Integer = 0 To numSteps - 1
            For j As Integer = 0 To faceList.Count - 1
                Dim moveTrans As Matrix = tg.CreateMatrix
                moveTrans.SetTranslation(faceList(j).normal)

                Dim graphicsTrans As Matrix = faceList(j).graphicsFace.Transformation
                graphicsTrans.TransformBy(moveTrans)
                faceList(j).graphicsFace.Transformation = graphicsTrans
            Next

            invApp.ActiveView.Update()
        Next

        If Not isForward Then
            oClientGraphics.Delete()
            body.Visible = True
        End If

        globalTrans.End()
    End Sub

    ' Used to graphically demonstrate the topology associated with a face.
    Public Sub ShowEdges(ByVal parentForm As frmBRepSamples)
        If invApp.ActiveDocumentType <> DocumentTypeEnum.kPartDocumentObject Then
            MsgBox("This must be run in a part document.")
        End If

        ' Have a face selected.
        Dim oFace As Face = Nothing
        Try
            oFace = invApp.CommandManager.Pick(SelectionFilterEnum.kPartFaceFilter, "Select a face")
        Catch ex As Exception
            Return
        End Try

        If oFace Is Nothing Then
            Return
        End If

        ' Enable the "Next" button on the dialog.
        parentForm.btnNext.Enabled = True

        Dim oPartDef As PartComponentDefinition = invApp.ActiveDocument.ComponentDefinition
        Dim oClientGraphics As ClientGraphics = oPartDef.ClientGraphicsCollection.Add("Edges")
        Dim node As GraphicsNode = oClientGraphics.AddNode(1)

        Dim text As TextGraphics = node.AddTextGraphics()
        text.Anchor = tg.CreatePoint(10, 40, 0)
        text.SetTransformBehavior(text.Anchor, DisplayTransformBehaviorEnum.kFrontFacingAndPixelScaling)
        text.SetViewSpaceAnchor(text.Anchor, tg.CreatePoint2d(10, 50), ViewLayoutEnum.kTopLeftViewCorner)
        text.FontSize = 60

        ' Create the face as client graphics.
        Dim surfGraphics As SurfaceGraphics = node.AddSurfaceGraphics(oFace)
        surfGraphics.DepthPriority = 8
        surfGraphics.Color = invApp.TransientObjects.CreateColor(255, 0, 0)
        text.Text = "Selected Face"
        invApp.ActiveView.Update()

        parentForm.doNext = False
        Do While Not parentForm.doNext
            System.Windows.Forms.Application.DoEvents()
        Loop

        Dim loopCount As Integer = 0
        For Each oLoop As EdgeLoop In oFace.EdgeLoops
            Dim edgeGraphics() As CurveGraphics
            ReDim edgeGraphics(oLoop.Edges.Count - 1)
            Dim edgeCount As Integer = 0
            For Each oEdge As Edge In oLoop.Edges
                edgeGraphics(edgeCount) = node.AddCurveGraphics(oEdge.Geometry)
                edgeGraphics(edgeCount).Color = invApp.TransientObjects.CreateColor(0, 255, 50)
                edgeGraphics(edgeCount).LineWeight = 3
                edgeGraphics(edgeCount).DepthPriority = 12
                edgeCount += 1
            Next

            text.Text = "Loop " & loopCount + 1 & " of " & oFace.EdgeLoops.Count
            loopCount += 1
            invApp.ActiveView.Update()
            parentForm.doNext = False
            Do While Not parentForm.doNext
                System.Windows.Forms.Application.DoEvents()
            Loop

            edgeCount = 0
            For Each oEdge As Edge In oLoop.Edges
                If oEdge.Faces.Count > 1 Then
                    edgeGraphics(edgeCount).Color = invApp.TransientObjects.CreateColor(255, 0, 255)
                    edgeGraphics(edgeCount).LineWeight = 10

                    text.Text = "Edge " & edgeCount + 1 & " of " & oLoop.Edges.Count & " in loop " & loopCount
                    invApp.ActiveView.Update()
                    parentForm.doNext = False
                    Do While Not parentForm.doNext
                        System.Windows.Forms.Application.DoEvents()
                    Loop

                    Dim fc As Face = Nothing
                    If oEdge.Faces.Item(1) Is oFace Then
                        fc = oEdge.Faces.Item(2)
                    Else
                        fc = oEdge.Faces.Item(1)
                    End If

                    Dim otherFace As SurfaceGraphics = node.AddSurfaceGraphics(fc)
                    otherFace.Color = invApp.TransientObjects.CreateColor(255, 255, 0)
                    otherFace.DepthPriority = 12

                    text.Text = "Connected Face to current edge"
                    invApp.ActiveView.Update()
                    parentForm.doNext = False
                    Do While Not parentForm.doNext
                        System.Windows.Forms.Application.DoEvents()
                    Loop

                    otherFace.Delete()

                    edgeGraphics(edgeCount).Color = invApp.TransientObjects.CreateColor(0, 255, 50)
                    edgeGraphics(edgeCount).LineWeight = 3

                    edgeCount += 1
                End If
            Next

            For i As Integer = 0 To edgeGraphics.Count - 1
                edgeGraphics(i).Delete()
            Next
        Next

        surfGraphics.Delete()

        oClientGraphics.Delete()
        invApp.ActiveView.Update()

        parentForm.btnNext.Enabled = False
    End Sub


    ' Displays a cone on each face of a selected body showing the normal of the face.
    Public Sub ShowBodyNormals()
        ' Get the active document.
        Dim doc As Document
        doc = invApp.ActiveDocument

        ' Check to see if client graphics already exist and delete them if they do.
        Dim graphics As ClientGraphics = Nothing
        Try
            graphics = doc.ComponentDefinition.ClientGraphicsCollection.Item("BodyNormalSample")
        Catch ex As Exception
        End Try

        If Not graphics Is Nothing Then
            graphics.Delete()
            invApp.ActiveView.Update()
            Exit Sub
        End If

        ' Have a body selected or if there's only one, use it.
        Dim Body As SurfaceBody = Nothing
        If doc.DocumentType = DocumentTypeEnum.kPartDocumentObject Then
            Dim partComp As PartComponentDefinition = doc.ComponentDefinition
            If partComp.SurfaceBodies.Count = 1 Then
                Body = partComp.SurfaceBodies.Item(1)
            Else
                Body = invApp.CommandManager.Pick(SelectionFilterEnum.kPartBodyFilter, "Select a body")
            End If
        Else
            Body = invApp.CommandManager.Pick(SelectionFilterEnum.kPartBodyFilter, "Select a body")
        End If

        ' Create client graphics.
        graphics = doc.ComponentDefinition.ClientGraphicsCollection.Add("BodyNormalSample")
        Dim node As GraphicsNode = graphics.AddNode(1)

        Dim transBRep As TransientBRep = invApp.TransientBRep
        Dim transGeom As TransientGeometry = invApp.TransientGeometry

        ' Determine the arrow size based on the overall range of the body.
        Dim arrowSize As Double = Body.RangeBox.MinPoint.DistanceTo(Body.RangeBox.MaxPoint) * 0.05

        ' Iterate over the faces of the body.
        Dim checkFace As Face
        For Each checkFace In Body.Faces
            ' Get a somewhat random point on the face.
            Dim testPoint As Point = checkFace.PointOnFace

            ' Get the normal of the face at a somewhat random point on the face.
            Dim normal As Vector = GetNormal(checkFace)
            normal.ScaleBy(arrowSize)

            ' Compute a point along the normal vector.
            Dim endPoint As Point = transGeom.CreatePoint(testPoint.X, testPoint.Y, testPoint.Z)
            endPoint.TranslateBy(normal)

            ' Create a transient body cone to represent the normal.
            Dim cone As SurfaceBody = transBRep.CreateSolidCylinderCone(testPoint, endPoint, arrowSize * 0.25, arrowSize * 0.25, 0)

            ' Create surface graphics of the body.
            Dim surfGraphics As SurfaceGraphics = node.AddSurfaceGraphics(cone)
            surfGraphics.Color = invApp.TransientObjects.CreateColor(200, 0, 0)
        Next

        invApp.ActiveView.Update()
    End Sub

    ' Draws a dense grid of normals over the selected face.
    Public Sub ShowFaceNormals(ByVal density As Integer)
        ' Get the active document.
        Dim doc As Document
        doc = invApp.ActiveDocument

        ' Check to see if client graphics already exist and delete them if they do.
        Dim graphics As ClientGraphics = Nothing
        Try
            graphics = doc.ComponentDefinition.ClientGraphicsCollection.Item("FaceNormalSample")
        Catch ex As Exception
        End Try

        If Not graphics Is Nothing Then
            graphics.Delete()
            doc.GraphicsDataSetsCollection.Item("FaceNormalSample").Delete()
            invApp.ActiveView.Update()
            Exit Sub
        End If

        ' Have a face selected.
        Dim oFace As Face = Nothing
        oFace = invApp.CommandManager.Pick(SelectionFilterEnum.kPartFaceFilter, "Select a face")

        Dim oEval As SurfaceEvaluator = oFace.Evaluator

        ' Determine the spacing between each sample point in parameter space
        Dim dMinU As Double = oEval.ParamRangeRect.MinPoint.X
        Dim dMaxU As Double = oEval.ParamRangeRect.MaxPoint.X
        Dim dMinV As Double = oEval.ParamRangeRect.MinPoint.Y
        Dim dMaxV As Double = oEval.ParamRangeRect.MaxPoint.Y
        Dim dUSpacing As Double = (dMaxU - dMinU) / (density - 1)
        Dim dVSpacing As Double = (dMaxV - dMinV) / (density - 1)

        ' Get the initial u and v values, which are the minimum values of the parametric range.
        Dim dCurrentU As Double = oEval.ParamRangeRect.MinPoint.X
        Dim dCurrentV As Double = oEval.ParamRangeRect.MinPoint.Y

        ' Define the array that will contain the u,v coordinates.
        Dim adUV() As Double
        ReDim adUV((density * density * 2) - 1)
        Dim lCurrentPoint As Long
        lCurrentPoint = 0
        Dim i As Long
        For i = 0 To density - 1
            Dim j As Long
            For j = 0 To density - 1
                ' Save the current u,v coordinate.
                adUV(lCurrentPoint * 2) = dCurrentU
                adUV(lCurrentPoint * 2 + 1) = dCurrentV

                ' Increment the u value and the current point being set.
                dCurrentU = dCurrentU + dUSpacing
                lCurrentPoint = lCurrentPoint + 1
            Next

            ' Set the u value back to the minimum and increment the v value.
            dCurrentU = oEval.ParamRangeRect.MinPoint.X
            dCurrentV = dCurrentV + dVSpacing
        Next

        ' Call the method on the SurfaceEvaluator object to calculate the normal vector for each
        ' of the points.   Also get the equivalent x,y,z coordinate for each parameter point.
        Dim adNormals() As Double = {}
        Call oEval.GetNormal(adUV, adNormals)
        Dim adNormalPoints() As Double = {}
        Call oEval.GetPointAtParam(adUV, adNormalPoints)

        ' Determine the normal size based on the overall range of the body.
        Dim arrowSize As Double = oFace.SurfaceBody.RangeBox.MinPoint.DistanceTo(oFace.SurfaceBody.RangeBox.MaxPoint) * 0.05

        Dim adCoords() As Double
        ReDim adCoords((density * density * 3 * 2) - 1)
        For i = 0 To (density * density) - 1
            adCoords(i * 6) = adNormalPoints(i * 3)
            adCoords(i * 6 + 1) = adNormalPoints(i * 3 + 1)
            adCoords(i * 6 + 2) = adNormalPoints(i * 3 + 2)
            adCoords(i * 6 + 3) = adNormalPoints(i * 3) + (adNormals(i * 3) * arrowSize)
            adCoords(i * 6 + 4) = adNormalPoints(i * 3 + 1) + (adNormals(i * 3 + 1) * arrowSize)
            adCoords(i * 6 + 5) = adNormalPoints(i * 3 + 2) + (adNormals(i * 3 + 2) * arrowSize)
        Next

        ' Create the client graphics objects.
        Dim iID As Integer
        Dim graphicsData As GraphicsDataSets = doc.GraphicsDataSetsCollection.Add("FaceNormalSample")
        Dim oCoords As GraphicsCoordinateSet
        iID = iID + 1
        oCoords = graphicsData.CreateCoordinateSet(iID)
        oCoords.PutCoordinates(adCoords)

        ' Create client graphics.
        graphics = doc.ComponentDefinition.ClientGraphicsCollection.Add("FaceNormalSample")
        Dim node As GraphicsNode = graphics.AddNode(1)

        iID += 1
        Dim oLineGraphics As LineGraphics = node.AddLineGraphics
        oLineGraphics.CoordinateSet = oCoords

        invApp.ActiveView.Update()
    End Sub

    ' Draws a grid over the face to illustrate the parametric space.
    Public Sub ShowParametricGrid(ByVal density As Integer)
        ' Get the active document.
        Dim doc As Document
        doc = invApp.ActiveDocument

        ' Check to see if client graphics already exist and delete them if they do.
        Dim graphics As ClientGraphics = Nothing
        Try
            graphics = doc.ComponentDefinition.ClientGraphicsCollection.Item("FaceParametricGridSample")
        Catch ex As Exception
        End Try

        If Not graphics Is Nothing Then
            graphics.Delete()
            doc.GraphicsDataSetsCollection.Item("FaceParametricGridSample").Delete()
            invApp.ActiveView.Update()
            Exit Sub
        End If

        ' Have a face selected.
        Dim oFace As Face = Nothing
        oFace = invApp.CommandManager.Pick(SelectionFilterEnum.kPartFaceFilter, "Select a face")

        Dim oEval As SurfaceEvaluator = oFace.Evaluator

        ' Set the number of samples that will be taken in the U and V directions.
        ' This will contol how smooth the curves looks.
        Dim sampleCount As Integer = 50

        ' Build up a list of all of the uv coordinates where we want to
        ' get the model coordinates.
        Dim surfEval As SurfaceEvaluator = oFace.Evaluator
        Dim paramMinPnt As Point2d = surfEval.ParamRangeRect.MinPoint
        Dim paramMaxPnt As Point2d = surfEval.ParamRangeRect.MaxPoint
        Dim adParams() As Double
        ReDim adParams((density * sampleCount * 4) - 1)
        Dim paramCount As Integer = 0

        ' Define the UV coordinates for the U curves.
        Dim currentU As Double = paramMinPnt.X
        Dim currentV As Double = paramMinPnt.Y
        Dim uStep As Double = (paramMaxPnt.X - paramMinPnt.X) / (density - 1)
        Dim vStep As Double = (paramMaxPnt.Y - paramMinPnt.Y) / (sampleCount - 1)
        For i As Integer = 0 To density - 1
            For j As Integer = 0 To sampleCount - 1
                adParams(paramCount * 2) = currentU
                adParams(paramCount * 2 + 1) = currentV
                currentV = currentV + vStep
                paramCount = paramCount + 1
            Next
            vStep = -vStep
            currentV = currentV + vStep
            currentU = currentU + uStep
        Next

        ' Define the UV coordinates for the V curves.
        currentU = paramMaxPnt.X
        currentV = paramMaxPnt.Y
        uStep = (paramMaxPnt.X - paramMinPnt.X) / (sampleCount - 1)
        vStep = (paramMaxPnt.Y - paramMinPnt.Y) / (density - 1)
        For i As Integer = 0 To density - 1
            For j As Integer = 0 To sampleCount - 1
                adParams(paramCount * 2) = currentU
                adParams(paramCount * 2 + 1) = currentV
                currentU = currentU - uStep
                paramCount = paramCount + 1
            Next
            uStep = -uStep
            currentV = currentV - vStep
            currentU = currentU - uStep
        Next

        Dim adCoords() As Double
        ReDim adCoords(((density * sampleCount * 3) + (density * sampleCount * 3)) - 1)
        Call surfEval.GetPointAtParam(adParams, adCoords)

        ' Create the client graphics objects.
        graphics = doc.ComponentDefinition.ClientGraphicsCollection.Add("FaceParametricGridSample")

        Dim iID As Integer = 1
        Dim oGraphicsData As GraphicsDataSets = doc.GraphicsDataSetsCollection.Add("FaceParametricGridSample")
        Dim oCoords As GraphicsCoordinateSet
        iID = iID + 1
        oCoords = oGraphicsData.CreateCoordinateSet(iID)
        oCoords.PutCoordinates(adCoords)

        Dim alIndices() As Integer
        ReDim alIndices((((((sampleCount - 1) * 2) * density) + ((sampleCount - 1) * 2) * density)) - 1)
        Dim lCurrentIndex As Integer = 1
        Dim k As Integer = 0
        For i = 0 To density - 1
            For j = 1 To sampleCount - 1
                k = k + 1
                alIndices((k * 2) - 2) = lCurrentIndex
                alIndices((k * 2) - 1) = lCurrentIndex + 1
                lCurrentIndex = lCurrentIndex + 1
            Next
            lCurrentIndex = lCurrentIndex + 1
        Next
        For i = 0 To density - 1
            For j = 1 To sampleCount - 1
                k = k + 1
                alIndices((k * 2) - 2) = lCurrentIndex
                alIndices(k * 2 - 1) = lCurrentIndex + 1
                lCurrentIndex = lCurrentIndex + 1
            Next
            lCurrentIndex = lCurrentIndex + 1
        Next

        iID = iID + 1
        Dim oIndexSet As GraphicsIndexSet = oGraphicsData.CreateIndexSet(iID)
        Call oIndexSet.PutIndices(alIndices)

        iID = iID + 1
        Dim oNode As GraphicsNode
        oNode = graphics.AddNode(iID)
        Dim oLineGraphics As LineGraphics
        oLineGraphics = oNode.AddLineGraphics
        oLineGraphics.CoordinateSet = oCoords
        oLineGraphics.CoordinateIndexSet = oIndexSet
        oLineGraphics.DepthPriority = 20

        invApp.ActiveView.Update()
    End Sub

    Public Sub ShowRangeBox()
        ' Get the active document and component
        Dim doc As Document = invApp.ActiveDocument
        Dim comp As ComponentDefinition = doc.ComponentDefinition

        Dim selectEnt As Object = Nothing
        If doc.SelectSet.Count > 0 Then
            selectEnt = doc.SelectSet.Item(1)
        End If

        ' Check to see if client graphics already exist and delete them if they do.
        Dim graphics As ClientGraphics = Nothing
        Try
            graphics = doc.ComponentDefinition.ClientGraphicsCollection.Item("BodyRangeBoxSample")
        Catch ex As Exception
        End Try

        If Not graphics Is Nothing Then
            graphics.Delete()
            doc.GraphicsDataSetsCollection.Item("BodyRangeBoxSample").Delete()
            invApp.ActiveView.Update()

            If selectEnt Is Nothing Then
                Exit Sub
            End If
        End If

        ' Get the first selected entity.
        If selectEnt Is Nothing Then
            MsgBox("The entity to display the range box for must be selected.")
            Exit Sub
        End If

        Dim boundBox As Box
        If TypeOf selectEnt Is Face Or TypeOf selectEnt Is Edge Then
            boundBox = selectEnt.Evaluator.RangeBox
        ElseIf TypeOf selectEnt Is ComponentOccurrence Then
            ' Get the range from the occurrence.
            Dim oOcc As ComponentOccurrence = selectEnt
            boundBox = oOcc.RangeBox
        ElseIf TypeOf selectEnt Is SurfaceBody Then
            Dim body As SurfaceBody = selectEnt
            boundBox = body.RangeBox
        Else
            MsgBox("The selected entity is not supported.")
            Exit Sub
        End If

        ' Create a graphics data set object.  This object contains all of the
        ' information used to define the graphics.
        Dim oDataSets As GraphicsDataSets
        oDataSets = doc.GraphicsDataSetsCollection.Add("BodyRangeBoxSample")

        ' Create a coordinate set.
        Dim oCoordSet As GraphicsCoordinateSet
        oCoordSet = oDataSets.CreateCoordinateSet(1)

        ' Create the client graphics for this compdef.
        Dim oClientGraphics As ClientGraphics = comp.ClientGraphicsCollection.Add("BodyRangeBoxSample")

        ' Create a graphics node.
        Dim oBoxNode As GraphicsNode = oClientGraphics.AddNode(1)
        oBoxNode.Selectable = False

        Dim oPointGraphics As PointGraphics = oBoxNode.AddPointGraphics
        oPointGraphics.PointRenderStyle = PointRenderStyleEnum.kFilledCirclePointStyle
        Dim pointCoords As GraphicsCoordinateSet = oDataSets.CreateCoordinateSet(2)
        pointCoords.Add(1, boundBox.MinPoint)
        pointCoords.Add(2, boundBox.MaxPoint)
        oPointGraphics.CoordinateSet = pointCoords

        ' Create line graphics.
        Dim oLineGraphics As LineGraphics = oBoxNode.AddLineGraphics

        oLineGraphics.CoordinateSet = oCoordSet

        Dim dBoxLines(71) As Double
        Dim minPoint(2) As Double
        Dim maxPoint(2) As Double
        boundBox.GetBoxData(minPoint, maxPoint)

        ' Line 1
        dBoxLines(0) = minPoint(0)
        dBoxLines(1) = minPoint(1)
        dBoxLines(2) = minPoint(2)
        dBoxLines(3) = maxPoint(0)
        dBoxLines(4) = minPoint(1)
        dBoxLines(5) = minPoint(2)

        ' Line 2
        dBoxLines(6) = minPoint(0)
        dBoxLines(7) = minPoint(1)
        dBoxLines(8) = minPoint(2)
        dBoxLines(9) = minPoint(0)
        dBoxLines(10) = maxPoint(1)
        dBoxLines(11) = minPoint(2)

        ' Line 3
        dBoxLines(12) = minPoint(0)
        dBoxLines(13) = minPoint(1)
        dBoxLines(14) = minPoint(2)
        dBoxLines(15) = minPoint(0)
        dBoxLines(16) = minPoint(1)
        dBoxLines(17) = maxPoint(2)

        ' Line 4
        dBoxLines(18) = maxPoint(0)
        dBoxLines(19) = maxPoint(1)
        dBoxLines(20) = maxPoint(2)
        dBoxLines(21) = minPoint(0)
        dBoxLines(22) = maxPoint(1)
        dBoxLines(23) = maxPoint(2)

        ' Line 5
        dBoxLines(24) = maxPoint(0)
        dBoxLines(25) = maxPoint(1)
        dBoxLines(26) = maxPoint(2)
        dBoxLines(27) = maxPoint(0)
        dBoxLines(28) = minPoint(1)
        dBoxLines(29) = maxPoint(2)

        ' Line 6
        dBoxLines(30) = maxPoint(0)
        dBoxLines(31) = maxPoint(1)
        dBoxLines(32) = maxPoint(2)
        dBoxLines(33) = maxPoint(0)
        dBoxLines(34) = maxPoint(1)
        dBoxLines(35) = minPoint(2)

        ' Line 7
        dBoxLines(36) = minPoint(0)
        dBoxLines(37) = maxPoint(1)
        dBoxLines(38) = minPoint(2)
        dBoxLines(39) = maxPoint(0)
        dBoxLines(40) = maxPoint(1)
        dBoxLines(41) = minPoint(2)

        ' Line 8
        dBoxLines(42) = minPoint(0)
        dBoxLines(43) = maxPoint(1)
        dBoxLines(44) = minPoint(2)
        dBoxLines(45) = minPoint(0)
        dBoxLines(46) = maxPoint(1)
        dBoxLines(47) = maxPoint(2)

        ' Line 9
        dBoxLines(48) = maxPoint(0)
        dBoxLines(49) = minPoint(1)
        dBoxLines(50) = maxPoint(2)
        dBoxLines(51) = maxPoint(0)
        dBoxLines(52) = minPoint(1)
        dBoxLines(53) = minPoint(2)

        ' Line 10
        dBoxLines(54) = maxPoint(0)
        dBoxLines(55) = minPoint(1)
        dBoxLines(56) = maxPoint(2)
        dBoxLines(57) = minPoint(0)
        dBoxLines(58) = minPoint(1)
        dBoxLines(59) = maxPoint(2)

        ' Line 11
        dBoxLines(60) = minPoint(0)
        dBoxLines(61) = minPoint(1)
        dBoxLines(62) = maxPoint(2)
        dBoxLines(63) = minPoint(0)
        dBoxLines(64) = maxPoint(1)
        dBoxLines(65) = maxPoint(2)

        ' Line 12
        dBoxLines(66) = maxPoint(0)
        dBoxLines(67) = minPoint(1)
        dBoxLines(68) = minPoint(2)
        dBoxLines(69) = maxPoint(0)
        dBoxLines(70) = maxPoint(1)
        dBoxLines(71) = minPoint(2)

        ' Assign the points into the coordinate set.
        Call oCoordSet.PutCoordinates(dBoxLines)

        ' Update the display.
        invApp.ActiveView.Update()
    End Sub

    Public Sub CreateTransBRep()
        ' Get the transient geometry object.
        Dim transBRep As TransientBRep = invApp.TransientBRep

        Dim transGeom As TransientGeometry = invApp.TransientGeometry

        ' Create a box.
        Dim boxDef As Inventor.Box = transGeom.CreateBox()
        boxDef.MinPoint = transGeom.CreatePoint(0, 0, 0)
        boxDef.MaxPoint = transGeom.CreatePoint(10, 8, 5)
        Dim boxBody As SurfaceBody = transBRep.CreateSolidBlock(boxDef)

        ' Create a cylinder.
        Dim cylBody As SurfaceBody = transBRep.CreateSolidCylinderCone(transGeom.CreatePoint(5, 4, 0),
                                                                       transGeom.CreatePoint(5, 4, 5),
                                                                       1, 1, 1)

        ' Subtract the cylinder from the box.
        transBRep.DoBoolean(boxBody, cylBody, BooleanTypeEnum.kBooleanTypeDifference)

        Dim partDoc As PartDocument = invApp.ActiveDocument
        Dim partComp As PartComponentDefinition = partDoc.ComponentDefinition

        partComp.Features.NonParametricBaseFeatures.Add(boxBody)
    End Sub
End Module
