﻿Imports Inventor

Module Corks
    ' Member variable declarations.
    Private Structure HoleInfo
        Dim position As Matrix
        Dim radius As Double
        Dim edge As EdgeProxy
    End Structure

    Private oDoc As AssemblyDocument
    Private OccurrencesList() As ComponentOccurrence
    Private LibraryPath As String

    ' Primary subroutine that inserts cords into all of the holes of the selected part.
    Public Sub InsertCorks()
        If invApp.ActiveDocumentType <> DocumentTypeEnum.kAssemblyDocumentObject Then
            MsgBox("An assembly must be open.")
            Exit Sub
        End If

        oDoc = invApp.ActiveDocument

        ' Have an occurrence selected.
        Dim occ As ComponentOccurrence
        occ = invApp.CommandManager.Pick(SelectionFilterEnum.kAssemblyOccurrenceFilter, "Select the occurrence.")

        If occ Is Nothing Then
            Exit Sub
        End If

        Dim HoleInfo() As HoleInfo = {}

        ' Call the function that examines the B-Rep and finds the holes.
        Call GetHoleInfo(HoleInfo, occ)

        ' Call the function to place the bolts using the previously obtained hole information.
        Dim trans As Transaction = Nothing
        Try
            trans = invApp.TransactionManager.StartTransaction(oDoc, "Place Corks")
            Call PlaceCorks(HoleInfo, oDoc)
        Catch ex As Exception
            trans.Abort()
            trans = Nothing
        End Try

        If Not trans Is Nothing Then
            trans.End()
        End If
    End Sub


    ' Utility function called by InsertCorks.  This does the work of actually placing the
    ' corks into the previously found holes.
    Private Sub PlaceCorks(HoleInfo() As HoleInfo, AssmDoc As AssemblyDocument)
        Dim oOccurrences As ComponentOccurrences
        oOccurrences = AssmDoc.ComponentDefinition.Occurrences
        Dim i As Integer
        For i = 0 To HoleInfo.Length - 1
            Dim Filename As String
            Filename = "Cork_" & Format(HoleInfo(i).radius, "0.000") & ".ipt"

            ' Check to see if a cork this size exists in the library directory.
            If Dir(g_CurrentPath & "Cork Library\" & Filename) = "" Then
                ' Create a new cork file of the correct size.

                ' Open the seed cork file.  (Can be opened visibly by setting the second
                ' argument to True.  This opens it invisibly for better performance.)
                Dim oCorkDoc As PartDocument
                oCorkDoc = invApp.Documents.Open(g_CurrentPath & "CorkSeed.ipt", False)

                ' Obtain a reference to the parameter called "Diameter".
                Dim DiamParam As Parameter
                DiamParam = oCorkDoc.ComponentDefinition.Parameters.Item("Diameter")

                ' Change the value of parameter, rounding it to 3 decimal places.
                DiamParam.Value = Val(Format(HoleInfo(i).radius * 2, "0.000"))

                ' Update the part.
                oCorkDoc.Update()

                ' Save the file using to the new filename.
                Call oCorkDoc.SaveAs(g_CurrentPath & "Cork Library\" & Filename, True)

                ' Close the seed cork file without saving.
                oCorkDoc.Close(True)
            End If

            ' Place the bolt using the matrix that was defined previously.
            Dim oCorkOccurrence As ComponentOccurrence
            oCorkOccurrence = oOccurrences.Add(g_CurrentPath & "Cork Library\" & Filename, HoleInfo(i).position)

            ' Get the edge of the bolt to use for the constraint.
            Dim oCorkEdge As EdgeProxy = Nothing
            oCorkEdge = GetCorkEdge(oCorkOccurrence)

            ' If an edge was defined, place a constraint.
            If Not oCorkEdge Is Nothing Then
                Dim oInsert As InsertConstraint
                oInsert = AssmDoc.ComponentDefinition.Constraints.AddInsertConstraint(oCorkEdge, HoleInfo(i).edge, False, 0)
            End If
        Next
    End Sub


    ' Given an occurrence cork, this uses the attribute that was previously placed
    ' in the cork to find an edge.
    Private Function GetCorkEdge(ByVal oOccurrence As ComponentOccurrence)
        ' Obtain the document the bolt occurrence references.
        Dim oDoc As PartDocument
        oDoc = oOccurrence.Definition.Document

        ' Query for the edge with the "AutoCork" attribute set.
        Dim oEdges As ObjectCollection
        oEdges = oDoc.AttributeManager.FindObjects("AutoCork")

        If oEdges.Count = 1 Then
            ' The edge obtained is in the context of the part document.
            ' Now get the edge in the context of this particular occurrence.
            Dim oEdge As Edge = Nothing
            Call oOccurrence.CreateGeometryProxy(oEdges.Item(1), oEdge)
            Return oEdge
        Else
            Return Nothing
        End If
    End Function

    ' Uses the B-Rep API to find "holes" in a part.  It looks for circular edges between
    ' planar faces and cylinders and assumes these are the edges of a hole.
    Private Sub GetHoleInfo(ByRef HoleInfo() As HoleInfo, Occurrence As ComponentOccurrence)
        ' Get the surface body of the first occurrence in the assembly
        Dim oOccurrenceBody As SurfaceBody
        oOccurrenceBody = Occurrence.SurfaceBodies(1)

        ' Iterate through loops of the body looking for loops containing
        ' single circular edges that connects a cylinder to a face that
        ' contains more than one loop.
        Dim oFace As Face
        For Each oFace In oOccurrenceBody.Faces
            If oFace.SurfaceType = SurfaceTypeEnum.kPlaneSurface Then
                Dim oLoop As EdgeLoop
                For Each oLoop In oFace.EdgeLoops
                    ' Check that this is an inner loop.  This will eliminate bosses and the bottoms of holes.
                    If Not oLoop.IsOuterEdgeLoop Then
                        ' Check to see if the loop contains a single circular edge.
                        If oLoop.Edges.Count = 1 And oLoop.Edges(1).CurveType = CurveTypeEnum.kCircleCurve Then
                            Dim oEdge As Edge = oLoop.Edges(1)

                            ' Get the faces joined by the edge.
                            Dim oCylinderFace As Face = Nothing
                            Dim oPlaneFace As Face = Nothing
                            If oEdge.Faces(1).SurfaceType = SurfaceTypeEnum.kCylinderSurface Then
                                oCylinderFace = oEdge.Faces(1)
                            ElseIf oEdge.Faces(1).SurfaceType = SurfaceTypeEnum.kPlaneSurface Then
                                oPlaneFace = oEdge.Faces(1)
                            End If

                            If oEdge.Faces(2).SurfaceType = SurfaceTypeEnum.kCylinderSurface Then
                                oCylinderFace = oEdge.Faces(2)
                            ElseIf oEdge.Faces(2).SurfaceType = SurfaceTypeEnum.kPlaneSurface Then
                                oPlaneFace = oEdge.Faces(2)
                            End If

                            If Not oCylinderFace Is Nothing And Not oPlaneFace Is Nothing Then
                                ' Check to see if the cylinder is perpendicular to the plane.
                                Dim oPlane As Plane = oPlaneFace.Geometry
                                Dim oCylinder As Cylinder = oCylinderFace.Geometry
                                If oPlane.Normal.IsParallelTo(oCylinder.AxisVector) Then
                                    ' Check to see if the normal of the cylinder points towards
                                    ' the axis of the cylinder or not.  If it does then it's a hole.
                                    ' If it doesn't then it is a boss.
                                    Dim oSurfEval As SurfaceEvaluator
                                    oSurfEval = oCylinderFace.Evaluator

                                    Dim pointArray(2) As Double
                                    Dim oPoint As Point = oCylinderFace.PointOnFace
                                    oPoint.GetPointData(pointArray)
                                    Dim adNormal(2) As Double
                                    oSurfEval.GetNormalAtPoint(pointArray, adNormal)

                                    ' Create a vector that's along the normal and has the same magnitude as the radius.
                                    Dim oNormal As Vector
                                    oNormal = tg.CreateVector(adNormal(0), adNormal(1), adNormal(2))
                                    oNormal.ScaleBy(oCylinder.Radius)

                                    ' Translate the point along the vector.
                                    oPoint.TranslateBy(oNormal)

                                    ' The point will now either be on the cylinder axis or way outside the cylinder.
                                    Dim oLine As Inventor.Line
                                    oLine = tg.CreateLine(oCylinder.BasePoint, oCylinder.AxisVector.AsVector)

                                    If WithinTol(oLine.DistanceTo(oPoint), 0, 0.00001) Then
                                        ' We have a hole, so now determine the hole information to be
                                        ' used when creating and placing the cork.

                                        ReDim Preserve HoleInfo(HoleInfo.Length)

                                        ' Save the edge.
                                        HoleInfo(HoleInfo.Length - 1).edge = oEdge

                                        Dim oCircle As Inventor.Circle
                                        oCircle = oEdge.Curve(oEdge.CurveType)

                                        ' Save the radius
                                        HoleInfo(HoleInfo.Length - 1).radius = oCircle.Radius

                                        ' Construct the matrix that will define the placement of the cork.
                                        Dim oOrigin As Point
                                        oOrigin = oCircle.Center

                                        ' Get a normal from the face.
                                        Dim oParam(1) As Double
                                        oParam(0) = oPlaneFace.Evaluator.ParamRangeRect.MinPoint.X
                                        oParam(1) = oPlaneFace.Evaluator.ParamRangeRect.MinPoint.Y
                                        Dim normal(2) As Double
                                        Call oPlaneFace.Evaluator.GetNormal(oParam, normal)

                                        Dim oZAxis As Vector
                                        oZAxis = tg.CreateVector(normal(0), normal(1), normal(2))

                                        ' Get a tangent from the face.
                                        Dim Tangent(2) As Double
                                        Call oPlaneFace.Evaluator.GetTangents(oParam, Tangent, Tangent)

                                        ' Set the X Vector of the matrix.
                                        Dim oXAxis As Vector
                                        oXAxis = tg.CreateVector(Tangent(0), Tangent(1), Tangent(2))

                                        ' Cross the Z and X to create the Y vector.
                                        Dim oYAxis As Vector
                                        oYAxis = oZAxis.CrossProduct(oXAxis)

                                        Dim oMatrix As Matrix
                                        oMatrix = tg.CreateMatrix
                                        oMatrix.SetCoordinateSystem(oOrigin, oXAxis, oYAxis, oZAxis)

                                        ' Save the matrix.
                                        HoleInfo(HoleInfo.Length - 1).position = oMatrix
                                    End If
                                End If
                            End If
                        End If
                    End If
                Next
            End If
        Next
    End Sub

    ' Utility function used to create the "AutoCork" attribute in the cork.  This only needs to be run
    ' once for a cork part and has already been used on the CorkSeed.ipt file delivered as part of
    ' this sample.
    Public Sub AddCorkAttribute()
        ' Make sure a part document is active.
        If invApp.ActiveDocumentType <> DocumentTypeEnum.kPartDocumentObject Then
            MsgBox("A circular edge in a part must be selected.")
            Exit Sub
        End If

        Dim oDoc As PartDocument = invApp.ActiveDocument

        Dim oEdge As Edge
        Try
            oEdge = invApp.ActiveDocument.SelectSet.Item(1)
        Catch ex As Exception
            MsgBox("A circular edge in a part must be selected.")
            Exit Sub
        End Try

        ' Check to see if the attribute set has already been used in this file.
        Dim oAttribSets As AttributeSetsEnumerator
        oAttribSets = oDoc.AttributeManager.FindAttributeSets("AutoCork")
        If oAttribSets.Count > 0 Then
            ' The attribute set already exists.  Highlight the existing object it's
            ' attached to.
            Dim oHS As HighlightSet = oDoc.HighlightSets.Add
            oHS.AddItem(oAttribSets.Item(1).Parent.Parent)

            ' Ask if they want to replace the existing attribute.
            If MsgBox("The highlighted edge has already been specified in this file.  Do you want to replace it?", vbQuestion + vbYesNo) = vbYes Then
                oHS.Clear()

                ' Delete the existing attribute.
                oAttribSets.Item(1).Delete()
            Else
                oHS.Clear()
                Exit Sub
            End If
        End If

        ' Add the attribute set to the selected edge.
        oEdge.AttributeSets.Add("AutoCork")
    End Sub

End Module
