﻿Imports Inventor

Module ModelPart
    Public Sub CreatePart()
        ' Create a new part document.
        Dim partDoc As PartDocument = invApp.Documents.Add(DocumentTypeEnum.kPartDocumentObject,
                                                           invApp.FileManager.GetTemplateFile(DocumentTypeEnum.kPartDocumentObject))

        ' Get the component of the part.
        Dim comp As PartComponentDefinition = partDoc.ComponentDefinition

        invApp.UserInterfaceManager.UserInteractionDisabled = True

        ' Create a sketch on the X-Y plane.
        Dim sk As PlanarSketch = comp.Sketches.Add(comp.WorkPlanes.Item(3))

        ' Draw the rectangle.
        sk.SketchLines.AddAsTwoPointRectangle(tg.CreatePoint2d(0, 0), tg.CreatePoint2d(2 * 2.54, 1.5 * 2.54))

        ' Create a profile.
        Dim prof As Profile = sk.Profiles.AddForSolid()

        ' Create the extrusion.
        Dim extrudeDef As ExtrudeDefinition = comp.Features.ExtrudeFeatures.CreateExtrudeDefinition(prof, PartFeatureOperationEnum.kNewBodyOperation)
        extrudeDef.SetDistanceExtent("0.2 in", PartFeatureExtentDirectionEnum.kPositiveExtentDirection)
        Dim extrudeOne As ExtrudeFeature = comp.Features.ExtrudeFeatures.Add(extrudeDef)

        ' Dim get the body that was just created.
        Dim body As SurfaceBody = extrudeOne.SurfaceBodies.Item(1)

        ' Find the back face of the extrusion. This finds it by coordinate position.
        Dim filter(0) As SelectionFilterEnum
        filter(0) = SelectionFilterEnum.kPartFaceFilter
        Dim foundObjs As ObjectsEnumerator
        foundObjs = comp.FindUsingPoint(tg.CreatePoint(1, 1.5 * 2.54, 0.1 * 2.54), filter, 0.01)

        ' Create a new sketch on the back face. This assumes one was found with FindUsingPoint.
        sk = comp.Sketches.Add(foundObjs.Item(1))

        ' Draw the shape on the new sketch. Because we don't know the orientation or position of the
        ' sketch, this uses model coordinates to specify the coordinates of the geometry.

        ' This starts by creating five points that will be used as input to create the sketch geometry
        ' in a point-to-point type of construction.
        Dim pnts(5) As SketchPoint
        Dim coord As Point = tg.CreatePoint(0.5 * 2.54, 1.5 * 2.54, 0.2 * 2.54)
        pnts(0) = sk.SketchPoints.Add(sk.ModelToSketchSpace(coord))

        coord.X = coord.X + (1 * 2.54)
        pnts(1) = sk.SketchPoints.Add(sk.ModelToSketchSpace(coord))

        coord.Z = coord.Z + (1 * 2.54)
        pnts(2) = sk.SketchPoints.Add(sk.ModelToSketchSpace(coord))

        coord.X = coord.X - (1 * 2.54)
        pnts(3) = sk.SketchPoints.Add(sk.ModelToSketchSpace(coord))

        coord.X = coord.X + (0.5 * 2.54)
        pnts(4) = sk.SketchPoints.Add(sk.ModelToSketchSpace(coord))

        Dim lines(2) As SketchLine
        lines(0) = sk.SketchLines.AddByTwoPoints(pnts(0), pnts(1))
        lines(1) = sk.SketchLines.AddByTwoPoints(pnts(1), pnts(2))
        lines(2) = sk.SketchLines.AddByTwoPoints(pnts(3), pnts(0))
        Dim arc As SketchArc = sk.SketchArcs.AddByCenterStartEndPoint(pnts(4), pnts(2), pnts(3), False)

        ' Create a profile.
        prof = sk.Profiles.AddForSolid()

        ' Create the extrusion.
        extrudeDef = comp.Features.ExtrudeFeatures.CreateExtrudeDefinition(prof, PartFeatureOperationEnum.kJoinOperation)
        extrudeDef.SetDistanceExtent("0.6 in", PartFeatureExtentDirectionEnum.kNegativeExtentDirection)
        Dim extrudeTwo As ExtrudeFeature = comp.Features.ExtrudeFeatures.Add(extrudeDef)

        ' Find the two edges as the front of the part to add fillets. This finds them by looking
        ' through all of the edges and find the two that have the correct coordinates for the vertices.
        Dim filletEdges As ObjectCollection = invApp.TransientObjects.CreateEdgeCollection
        For Each oEdge As Edge In body.Edges
            If WithinTol(oEdge.StartVertex.Point.X, 0) Then
                If WithinTol(oEdge.StopVertex.Point.X, 0) Then
                    If WithinTol(oEdge.StartVertex.Point.Y, 0) Then
                        If WithinTol(oEdge.StopVertex.Point.Y, 0) Then
                            filletEdges.Add(oEdge)
                        End If
                    End If
                End If
            ElseIf WithinTol(oEdge.StartVertex.Point.X, 2 * 2.54) Then
                If WithinTol(oEdge.StopVertex.Point.X, 2 * 2.54) Then
                    If WithinTol(oEdge.StartVertex.Point.Y, 0) Then
                        If WithinTol(oEdge.StopVertex.Point.Y, 0) Then
                            filletEdges.Add(oEdge)
                        End If
                    End If
                End If
            End If

            ' Exit out of the for loop if both edges have been found.
            If filletEdges.Count = 2 Then
                Exit For
            End If
        Next

        ' Fillet the edges.
        Dim cornerFillet As FilletFeature = comp.Features.FilletFeatures.AddSimple(filletEdges, "0.4 in")

        ' Find the edges between the base and the tombstone that need to be filleted.
        ' This illustrates this by finding the common edges between the top face of
        ' the first extrusion and the side faces of the second extrusion.
        filletEdges = invApp.TransientObjects.CreateEdgeCollection
        Dim topFace As Face = extrudeOne.EndFaces.Item(1)
        For Each sideFace As Face In extrudeTwo.SideFaces
            ' Check to see if any of the edges connect to the top face of the first extrusion.
            For Each checkEdge As Edge In sideFace.Edges
                For Each connectedFace As Face In checkEdge.Faces
                    If connectedFace Is topFace Then
                        filletEdges.Add(checkEdge)
                    End If
                Next
            Next
        Next

        ' Fillet the edges.
        comp.Features.FilletFeatures.AddSimple(filletEdges, "0.25 in")

        ' Find the cylindrical side face of the tombstone extrusion to create a concentric hole.
        Dim concentric As ConcentricHolePlacementDefinition
        For Each checkFace As Face In extrudeTwo.SideFaces
            If checkFace.SurfaceType = SurfaceTypeEnum.kCylinderSurface Then
                concentric = comp.Features.HoleFeatures.CreateConcentricPlacementDefinition(extrudeTwo.EndFaces.Item(1), checkFace)
                comp.Features.HoleFeatures.AddDrilledByThroughAllExtent(concentric, "0.4 in", PartFeatureExtentDirectionEnum.kPositiveExtentDirection)
            End If
        Next

        ' Use the cylindrical faces from the fillets to create the other holes.
        concentric = comp.Features.HoleFeatures.CreateConcentricPlacementDefinition(extrudeOne.StartFaces.Item(1), cornerFillet.Faces.Item(1))
        comp.Features.HoleFeatures.AddDrilledByThroughAllExtent(concentric, "0.4 in", PartFeatureExtentDirectionEnum.kPositiveExtentDirection)

        concentric = comp.Features.HoleFeatures.CreateConcentricPlacementDefinition(extrudeOne.StartFaces.Item(1), cornerFillet.Faces.Item(2))
        comp.Features.HoleFeatures.AddDrilledByThroughAllExtent(concentric, "0.4 in", PartFeatureExtentDirectionEnum.kPositiveExtentDirection)

        invApp.UserInterfaceManager.UserInteractionDisabled = False

        Dim cam As Camera = invApp.ActiveView.Camera
        cam.ViewOrientationType = ViewOrientationTypeEnum.kBottomViewOrientation
        cam.ApplyWithoutTransition()
        invApp.ActiveView.SetCurrentAsFront()

        cam.ViewOrientationType = ViewOrientationTypeEnum.kIsoTopRightViewOrientation
        cam.Fit()
        cam.Apply()
    End Sub
End Module
